﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>Request to execute a scheduled managed action immediately.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/ApplyEnvironmentManagedActionRequest">AWS
 * API Reference</a></p>
 */
class ApplyEnvironmentManagedActionRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API ApplyEnvironmentManagedActionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ApplyEnvironmentManagedAction"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the target environment.</p>
   */
  inline const Aws::String& GetEnvironmentName() const { return m_environmentName; }
  inline bool EnvironmentNameHasBeenSet() const { return m_environmentNameHasBeenSet; }
  template <typename EnvironmentNameT = Aws::String>
  void SetEnvironmentName(EnvironmentNameT&& value) {
    m_environmentNameHasBeenSet = true;
    m_environmentName = std::forward<EnvironmentNameT>(value);
  }
  template <typename EnvironmentNameT = Aws::String>
  ApplyEnvironmentManagedActionRequest& WithEnvironmentName(EnvironmentNameT&& value) {
    SetEnvironmentName(std::forward<EnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The environment ID of the target environment.</p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  ApplyEnvironmentManagedActionRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action ID of the scheduled managed action to execute.</p>
   */
  inline const Aws::String& GetActionId() const { return m_actionId; }
  inline bool ActionIdHasBeenSet() const { return m_actionIdHasBeenSet; }
  template <typename ActionIdT = Aws::String>
  void SetActionId(ActionIdT&& value) {
    m_actionIdHasBeenSet = true;
    m_actionId = std::forward<ActionIdT>(value);
  }
  template <typename ActionIdT = Aws::String>
  ApplyEnvironmentManagedActionRequest& WithActionId(ActionIdT&& value) {
    SetActionId(std::forward<ActionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_environmentName;

  Aws::String m_environmentId;

  Aws::String m_actionId;
  bool m_environmentNameHasBeenSet = false;
  bool m_environmentIdHasBeenSet = false;
  bool m_actionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
