﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/mediaconvert/MediaConvertRequest.h>
#include <aws/mediaconvert/MediaConvert_EXPORTS.h>
#include <aws/mediaconvert/model/AccelerationSettings.h>
#include <aws/mediaconvert/model/HopDestination.h>
#include <aws/mediaconvert/model/JobTemplateSettings.h>
#include <aws/mediaconvert/model/StatusUpdateInterval.h>

#include <utility>

namespace Aws {
namespace MediaConvert {
namespace Model {

/**
 */
class CreateJobTemplateRequest : public MediaConvertRequest {
 public:
  AWS_MEDIACONVERT_API CreateJobTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateJobTemplate"; }

  AWS_MEDIACONVERT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * Accelerated transcoding can significantly speed up jobs with long, visually
   * complex content. Outputs that use this feature incur pro-tier pricing. For
   * information about feature limitations, see the AWS Elemental MediaConvert User
   * Guide.
   */
  inline const AccelerationSettings& GetAccelerationSettings() const { return m_accelerationSettings; }
  inline bool AccelerationSettingsHasBeenSet() const { return m_accelerationSettingsHasBeenSet; }
  template <typename AccelerationSettingsT = AccelerationSettings>
  void SetAccelerationSettings(AccelerationSettingsT&& value) {
    m_accelerationSettingsHasBeenSet = true;
    m_accelerationSettings = std::forward<AccelerationSettingsT>(value);
  }
  template <typename AccelerationSettingsT = AccelerationSettings>
  CreateJobTemplateRequest& WithAccelerationSettings(AccelerationSettingsT&& value) {
    SetAccelerationSettings(std::forward<AccelerationSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Optional. A category for the job template you are creating
   */
  inline const Aws::String& GetCategory() const { return m_category; }
  inline bool CategoryHasBeenSet() const { return m_categoryHasBeenSet; }
  template <typename CategoryT = Aws::String>
  void SetCategory(CategoryT&& value) {
    m_categoryHasBeenSet = true;
    m_category = std::forward<CategoryT>(value);
  }
  template <typename CategoryT = Aws::String>
  CreateJobTemplateRequest& WithCategory(CategoryT&& value) {
    SetCategory(std::forward<CategoryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Optional. A description of the job template you are creating.
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateJobTemplateRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Optional. Use queue hopping to avoid overly long waits in the backlog of the
   * queue that you submit your job to. Specify an alternate queue and the maximum
   * time that your job will wait in the initial queue before hopping. For more
   * information about this feature, see the AWS Elemental MediaConvert User Guide.
   */
  inline const Aws::Vector<HopDestination>& GetHopDestinations() const { return m_hopDestinations; }
  inline bool HopDestinationsHasBeenSet() const { return m_hopDestinationsHasBeenSet; }
  template <typename HopDestinationsT = Aws::Vector<HopDestination>>
  void SetHopDestinations(HopDestinationsT&& value) {
    m_hopDestinationsHasBeenSet = true;
    m_hopDestinations = std::forward<HopDestinationsT>(value);
  }
  template <typename HopDestinationsT = Aws::Vector<HopDestination>>
  CreateJobTemplateRequest& WithHopDestinations(HopDestinationsT&& value) {
    SetHopDestinations(std::forward<HopDestinationsT>(value));
    return *this;
  }
  template <typename HopDestinationsT = HopDestination>
  CreateJobTemplateRequest& AddHopDestinations(HopDestinationsT&& value) {
    m_hopDestinationsHasBeenSet = true;
    m_hopDestinations.emplace_back(std::forward<HopDestinationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The name of the job template you are creating.
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateJobTemplateRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Specify the relative priority for this job. In any given queue, the service
   * begins processing the job with the highest value first. When more than one job
   * has the same priority, the service begins processing the job that you submitted
   * first. If you don't specify a priority, the service uses the default value 0.
   */
  inline int GetPriority() const { return m_priority; }
  inline bool PriorityHasBeenSet() const { return m_priorityHasBeenSet; }
  inline void SetPriority(int value) {
    m_priorityHasBeenSet = true;
    m_priority = value;
  }
  inline CreateJobTemplateRequest& WithPriority(int value) {
    SetPriority(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Optional. The queue that jobs created from this template are assigned to. If you
   * don't specify this, jobs will go to the default queue.
   */
  inline const Aws::String& GetQueue() const { return m_queue; }
  inline bool QueueHasBeenSet() const { return m_queueHasBeenSet; }
  template <typename QueueT = Aws::String>
  void SetQueue(QueueT&& value) {
    m_queueHasBeenSet = true;
    m_queue = std::forward<QueueT>(value);
  }
  template <typename QueueT = Aws::String>
  CreateJobTemplateRequest& WithQueue(QueueT&& value) {
    SetQueue(std::forward<QueueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * JobTemplateSettings contains all the transcode settings saved in the template
   * that will be applied to jobs created from it.
   */
  inline const JobTemplateSettings& GetSettings() const { return m_settings; }
  inline bool SettingsHasBeenSet() const { return m_settingsHasBeenSet; }
  template <typename SettingsT = JobTemplateSettings>
  void SetSettings(SettingsT&& value) {
    m_settingsHasBeenSet = true;
    m_settings = std::forward<SettingsT>(value);
  }
  template <typename SettingsT = JobTemplateSettings>
  CreateJobTemplateRequest& WithSettings(SettingsT&& value) {
    SetSettings(std::forward<SettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Specify how often MediaConvert sends STATUS_UPDATE events to Amazon CloudWatch
   * Events. Set the interval, in seconds, between status updates. MediaConvert sends
   * an update at this interval from the time the service begins processing your job
   * to the time it completes the transcode or encounters an error.
   */
  inline StatusUpdateInterval GetStatusUpdateInterval() const { return m_statusUpdateInterval; }
  inline bool StatusUpdateIntervalHasBeenSet() const { return m_statusUpdateIntervalHasBeenSet; }
  inline void SetStatusUpdateInterval(StatusUpdateInterval value) {
    m_statusUpdateIntervalHasBeenSet = true;
    m_statusUpdateInterval = value;
  }
  inline CreateJobTemplateRequest& WithStatusUpdateInterval(StatusUpdateInterval value) {
    SetStatusUpdateInterval(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * The tags that you want to add to the resource. You can tag resources with a
   * key-value pair or with only a key.
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateJobTemplateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateJobTemplateRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  AccelerationSettings m_accelerationSettings;

  Aws::String m_category;

  Aws::String m_description;

  Aws::Vector<HopDestination> m_hopDestinations;

  Aws::String m_name;

  int m_priority{0};

  Aws::String m_queue;

  JobTemplateSettings m_settings;

  StatusUpdateInterval m_statusUpdateInterval{StatusUpdateInterval::NOT_SET};

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_accelerationSettingsHasBeenSet = false;
  bool m_categoryHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_hopDestinationsHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_priorityHasBeenSet = false;
  bool m_queueHasBeenSet = false;
  bool m_settingsHasBeenSet = false;
  bool m_statusUpdateIntervalHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaConvert
}  // namespace Aws
